/* Processing code.

   Cygnus IRC Services - Copyright (c) 2001-2002 Darcy Grexton
   Contact: skold@habber.net, skold @ HabberNet

   See doc/LICENSE for licensing details.
 */

#include "../inc/services.h"

/* split_buf:  Split a buffer into arguments and store the arguments in an
               argument vector pointed to by argv (which will be malloc'd
               as necessary); return the argument count.  If colon_special
               is non-zero, then treat a parameter with a leading ':' as
               the last parameter of the line, per the IRC RFC. Destroys
               the buffer by side effect.
 */
int8 split_buf (char *buf, char ***argv)
{
    int8 argvsize = 8;
    int8 argc;
    char *s;

    *argv = smalloc (sizeof(char *) * argvsize);
    argc = 0;

    while (*buf)
    {
        if (argc == argvsize)
        {
            argvsize += 8;
            *argv = srealloc (*argv, sizeof(char *) * argvsize);
        }
        if (*buf == ':')
        {
            (*argv)[argc++] = buf+1;
            buf = "";
        }
        else
        {
            s = strpbrk(buf, " ");
            if (s)
            {
                *s++ = 0;
                while (isspace(*s))
                    s++;
            }
            else
                s = buf + strlen(buf);
            (*argv)[argc++] = buf;
            buf = s;
       }
    }
     
    return argc;
}

/* process:  Main processing routine. */
void process ()
{
    char source[64];
    char cmd[64];
    char buf[512];
    char *s;
    int8 ac;
    char **av;
    Message *m;

    /* If debugging, log the buffer. */
    if (debuglevel == 1 || debuglevel == 3)
	debug ("--> %s", inbuf);

    /* First make a copy of the buffer so we have the original in case we
       crash - in that case, we want to know what we crashed on.
     */
    strscpy (buf, inbuf, sizeof (buf));

    /* Split the buffer into pieces. */
    if (*buf == ':')
    {
	s = strpbrk (buf, " ");

	if (!s)
	    return;
	*s = 0;

	while (isspace(*++s))
	    ;

	strscpy (source, buf + 1, sizeof (source));
	memmove (buf, s, strlen (s) + 1);
    }
    else
	*source = 0;

    if (!*buf)
	return;

    s = strpbrk (buf, " ");

    if (s)
    {
	*s = 0;
	while (isspace (*s++))
	    ;
    }
    else
	s = buf + strlen(buf);

    strscpy (cmd, buf, sizeof(cmd));
    ac = split_buf (s, &av);

    /* Do something with the message. */
    m = find_message (cmd);

    if (m)
	if (m->func)
	    m->func (source, ac, av);

    /* Free argument list we created */
    free (av);
}

/* Send our MOTD */
void m_motd (char *source, int8 ac, char **av)
{
    NickInfo *ni;
    SRA *sra;
    User *u;
    FILE *f;
    char buf[BUFSIZE];
    int uptime = time (NULL) - me.since;

    send_cmd (me.name, "375 %s :- %s Message of the Day -", source, me.name);
    send_cmd (me.name, "372 %s :- ", source);
    send_cmd (me.name,
	"372 %s :- Cygnus IRC Services v%s Copyright (c) 2001-2002 Darcy Grexton",
	source, version);
    send_cmd (me.name, "372 %s :- ", source);

    f = fopen (MOTDFILE, "r");

    if (f)
    {
	while (fgets (buf, sizeof (buf), f))
	{
	    buf[strlen (buf) - 1] = 0;
	    send_cmd (me.name, "372 %s :- %s", source, buf);
	}
	fclose (f);
    }

    /* List the csops. Ones who are online should be in bold. */
    send_cmd (me.name, "372 %s :- ", source);
    send_cmd (me.name, "372 %s :- Channels expire after \2%s\2 of non use.", source,
	duration (chan_expire, 2));
    send_cmd (me.name, "372 %s :- Nicknames expire after \2%s\2 of non use.", source,
	duration (nick_expire, 2));
    send_cmd (me.name, "372 %s :- ", source);
    send_cmd (me.name, "372 %s :- The following people can help you recover lost or", source);
    send_cmd (me.name, "372 %s :- forgotten passwords. \2Bold\2 nicknames are online.", source);
    send_cmd (me.name, "372 %s :- ", source);

    *buf = 0;

    for (sra = sralist; sra; sra = sra->next)
    {
	ni = findnick (sra->nick);

	if (ni)
	{
	    if ((u = finduser (ni->nick)) && is_oper (u))
	    {
		strcat (buf, "\2");
		strcat (buf, ni->nick);
		strcat (buf, "\2");
	    }
	    else
		strcat (buf, ni->nick);

	    strcat (buf, " ");

	    if (strlen (buf) > 70)
	    {
		send_cmd (me.name, "372 %s :- %s", source, buf);
		*buf = 0;
	    }
	}
    }

    for (ni = firstni (); ni; ni = nextni ())
    {
	if (can_csop (ni->nick) && !can_sra (ni->nick))
	{
	    if ((u = finduser (ni->nick)) && is_oper (u))
	    {
		strcat (buf, "\2");
		strcat (buf, ni->nick);
		strcat (buf, "\2");
	    }
	    else
		strcat (buf, ni->nick);

	    strcat (buf, " ");

	    if (strlen (buf) > 70)
	    {
		send_cmd (me.name, "372 %s :- %s", source, buf);
		*buf = 0;
	    }
	}
    }

    if (*buf)
	send_cmd (me.name, "372 %s :- %s", source, buf);

    send_cmd (me.name, "372 %s :- ", source);
    send_cmd (me.name, "372 %s :- Current uptime: %d days, %d:%02d:%02d", source, uptime/86400,
        (uptime/3600) % 24, (uptime/60) % 60, uptime % 60);
    send_cmd (me.name, "372 %s :- Highest uptime: %d days, %d:%02d:%02d", source, maxuptime/86400,
	(maxuptime/3600) % 24, (maxuptime/60) % 60, maxuptime % 60);

    send_cmd (me.name, "372 %s :- ", source);
    send_cmd (me.name, "376 %s :End of /MOTD command.", source);
}

/* Reply to a TIME query */
void m_time (char *source, int8 ac, char **av)
{
    time_t t;
    struct tm *tm;
    char buf[80];

    time (&t);
    tm = localtime (&t);
    strftime (buf, sizeof (buf), "%A %B %d %Y -- %H:%M %z", tm);

    send_cmd (me.name, "391 %s %s :%s", source, me.name, buf);
}

/* Send an ADMIN reply */
void m_admin (char *source, int8 ac, char **av)
{
    send_cmd (me.name, "256 %s :Administrative info about %s", source,
	me.name);
    send_cmd (me.name, "257 %s :Cygnus IRC Services v%s", source, version);
    send_cmd (me.name, "258 %s :Copyright (c) 2001-2002 Darcy Grexton",
	source);
    send_cmd (me.name,
	"259 %s :For more information E-Mail skold@habber.net.", source);
}

/* Handle an ERROR command */
void m_error (char *source, int8 ac, char **av)
{
    snprintf (quitmsg, BUFSIZE, "%s", inbuf);
    log (RPL_ERROR, inbuf);
    runflags |= RUN_SHUTDOWN;
}

/* Reply to a ping */
void m_ping (char *source, int8 ac, char **av)
{
    if (ac < 1)
	return;

    /* Reply to the ping with the servername the ping came for. This
       stops bad ping replies when we get PINGs for things like jupes.
       Bahamut seems to fuck up pings when bursting though, so we have
       to account for that too..
     */
    if (ac > 1)
	send_cmd (av[1], "%s %s %s", me.token ? "9" : "PONG",
	    ac > 1 ? av[1] : me.name, av[0]);
    else
	send_cmd (me.name, "%s %s %s", me.token ? "9" : "PONG",
	    ac > 1 ? av[1] : me.name, av[0]);

    if (ircdtype == BAHAMUT)
    {
	/* Bahamut sends two PINGs at the end of the burst. We'll use that to figure
	   out when it's over.
	 */
	if (bursting > 0)
	    bursting--;

	/* Now that the burst is done, run through every channel, and do the modechanges
	   and topic changes as needed.
	 */
	if (bursted && !bursting)
	{
	    globops (s_RootServ, "Synched to network in %s: %d users, %d channels, %d servers.",
		duration (time (NULL) - burststart, 1), userstat, chancnt, servcnt + 1);

#ifdef HAVE_GETTIMEOFDAY
	    gettimeofday (&burstnow, NULL);
	    timersub (&burstnow, &burstms, &bursttmp);

	    if (tv2ms (&bursttmp) > 1000)
		log_sameline (0, "completed in %s.", duration (tv2ms (&bursttmp) / 1000, 1));
	    else
		log_sameline (0, "completed in %d ms.", tv2ms (&bursttmp));
#endif

	    check_eb ();
	}

	/* Since Bahamut uses PINGs to show the end of the burst, we need somethign different
	   to track the end of burst. Otherwise we'll keep sending sync'd notices with each
	   ping.
	 */
	if (bursting == 0)
	    bursted = 0;
    }
}

/* Handle a KICK command */
void m_kick (char *source, int8 ac, char **av)
{
    if (ac != 3)
	return; 

    do_kick (source, ac, av);
}

/* Handle a TOPIC command */
void m_topic (char *source, int8 ac, char **av)
{
    if (ac != 4)
	return;

    do_topic (source, ac, av);
}

/* Handle a JOIN command */
void m_join (char *source, int8 ac, char **av)
{
    if (ac != 1)
	return;

    do_join (source, ac, av);
}

/* Handle an SJOIN command. */
void m_sjoin (char *source, int8 ac, char **av)
{
    do_sjoin (source, ac, av);
}

/* Handle a PART command */
void m_part (char *source, int8 ac, char **av)
{
    if (ac < 1 || ac > 2)   
	return;

    do_part (source, ac, av);
}

/* Handle a NICK command */
void m_nick (char *source, int8 ac, char **av)
{
    do_nick (source, ac, av);
}

/* Handle a QUIT command */
void m_quit (char *source, int8 ac, char **av)
{
    if (ac != 1)
	return;

    if (!finduser (source))
    {
	if (debuglevel > 1)
	    debug ("do_quit(): Got QUIT for nonexistant user %s", source);

	return;
    }

    do_quit (source, ac, av);
}

/* Send our version reply */
void m_version (char *source, int8 ac, char **av)
{
    if (source)
    {
	/* Send version replies with the server name that was queried. If it's
	   not the same as our name, theyre likely versioning a juped server,
	   so we'll reply as such.
	 */
	if (!stricmp (me.name, av[0]))
	    send_cmd (me.name, "351 %s cygnus-%s. %s :%s%s%s%s%s%s%s",
		source, version, me.name, (rootserv_on == TRUE) ? "R" : "",
		(nickserv_on == TRUE) ? "N" : "", (chanserv_on == TRUE) ? "C" : "",
		(memoserv_on == TRUE) ? "M" : "", (globalnoticer_on == TRUE) ? "g" : "",
#ifdef WIN32
		"W",
#else
		"",
#endif
		debuglevel ? "d" : "");
	else
	    send_cmd (av[0], "351 %s juped-server. %s :XX", source, av[0]);
    }
}

/* A server replied to a version query. Was it stats? */
void m_vreply (char *source, int8 ac, char **av)
{
    /* Intercept SVSSEND */
    if (!stricmp (av[1], "0"))
    {
	if (!stricmp (source, operstats_server))
	{
	    char *cmd = strtok (av[2], " "), *param = strtok (NULL, "");

	    if (!cmd || !param)
		return;

	    /* AKILL */
	    if (!stricmp (cmd, "1"))
	    {
		AutoKill *akill;
		char *mask = strtok (param, " "), *setter = strtok (NULL, " ");
		char *realname = strtok (NULL, " "), *set = strtok (NULL, " ");
		char *expires = strtok (NULL, " "), *reason = strtok (NULL, "");

		if (!mask || !setter || !realname || !set || !expires || !reason)
		    return;

		/* Ignore it if we already have it. */
		if (is_akilled (mask))
		    return;

		if (akillcnt >= akill_size)
		{
		    if (akill_size < 8)
			akill_size = 8;
		    else if (akill_size >= 16384)
			akill_size = 32767;
		    else
			akill_size *= 2;

		    akills = srealloc (akills, sizeof (*akills) * akill_size);
		}

		akill = &akills[akillcnt];
		akill->mask = sstrdup (mask);
		akill->reason = sstrdup (reason);
		akill->setter = sstrdup (setter);
		akill->realname = atoi (realname);
		akill->set = atol (set);
		akill->expires = atoi (expires);

		akillcnt++;

		return;
	    }

	    /* RAKILL */
	    if (!stricmp (cmd, "2"))
	    {
		char *mask = strtok (param, " ");
		int16 i;

 		/* Make sure we have it */
		if (!is_akilled (mask))
		    return;

		for (i = 0; i < akillcnt && stricmp (akills[i].mask, mask); i++);

		if (i < akillcnt)
		{
		    free (akills[i].mask);
		    free (akills[i].reason);
		    free (akills[i].setter);

		    akillcnt--;

		    if (i < akillcnt)
			memmove (akills + i, akills + i + 1, sizeof (*akills) * (akillcnt - i));
		}

		return;
	    }

	    /* AUTH */
	    if (!stricmp (cmd, "3"))
	    {
		User *u = finduser (param);

		if (u)
		    u->flags |= UF_SRA;

		return;
	    }

	    /* DEAUTH */
	    if (!stricmp (cmd, "4"))
	    {
		User *u = finduser (param);

		if (u)
		    u->flags &= ~UF_SRA;

		return;
	    }
	}
	else
	    globops (s_RootServ, "Received SVSSEND from unexpected server %s, please check "
		"your configuration.", source);
    }
}

/* Handle an AWAY command */
void m_away (char *source, int8 ac, char **av)
{
    User *u = finduser (source);

    if (u && (ac == 0 || *av[0] == 0))
    {
	check_memos (u);

	if (is_csop (u))
	    check_verifies (u);
    }
}

/* Reply to an INFO command */
void m_info (char *source, int8 ac, char **av)
{
    int i;

    for (i = 0; infotext[i]; i++)
	send_cmd (me.name, "371 %s :%s", source, infotext[i]);
    send_cmd (me.name, "371 %s :cygnus-%s", source, version);
    send_cmd (me.name,
	      "371 %s :Birth Date: %s, compile # %s", source, compile_time,
	      build);
    send_cmd (me.name, "371 %s :On-line since %s", source, ctime (&me.since));
    send_cmd (me.name, "374 %s :End of /INFO list.", source);
}

/* Reply to a NETINFO command. */
void m_netinfo (char *source, int8 ac, char **av)
{
    send_cmd (NULL, "%s %d %lu 0 %s * * * :%s", me.token ? "AO" : "NETINFO",
	usercnt, time (NULL), av[3], av[7]);

    globops (s_RootServ, "Synched to network in %s: %d users, %d channels, %d servers.",
	duration (time (NULL) - burststart, 1), userstat, chancnt, servcnt + 1);

#ifdef HAVE_GETTIMEOFDAY
    gettimeofday (&burstnow, NULL);
    timersub (&burstnow, &burstms, &bursttmp);

    if (tv2ms (&bursttmp) > 1000)
	log_sameline (0, "completed in %s.", duration (tv2ms (&bursttmp) / 1000, 1));
    else
	log_sameline (0, "completed in %d ms.", tv2ms (&bursttmp));
#endif

    bursting = 0;

    /* Now that the burst is done, run through every channel, and do the modechanges
       and topic changes as needed.
     */
    check_eb ();
}

/* Reply to a STATS query */
void m_stats (char *source, int8 ac, char **av)
{
    if (ac < 1)
	return;

    switch (*av[0])
    {
	case 'u':
	{
	    int uptime = time (NULL) - me.since;
	    int tmp = servcount ();

	    send_cmd (me.name, "242 %s :Server Up %d days, %d:%02d:%02d",
		source, uptime/86400, (uptime/3600) % 24, (uptime/60) % 60,
		uptime % 60);
	    send_cmd (me.name,
		"250 %s :Highest connection count: %d (%d clients)",
		source, tmp + 1, tmp);
	    send_cmd (me.name, "219 %s u :End of /STATS report.", source);
	    break;
	}
	default:
	    send_cmd (me.name, "219 %s * :End of /STATS report.", source);
    }
}

/* Reply to a WHOIS query */
void m_whois (char *source, int8 ac, char **av)
{   
    if (ac < 1)
	return;   

    if ((!stricmp (av[0], s_RootServ)) && rootserv_on == TRUE)
    {
	send_cmd (me.name, "311 %s %s %s %s * :%s", source, s_RootServ,
	    rs.user, rs.host, rs.real);
	send_cmd (me.name, "312 %s %s %s :%s", source, s_RootServ,
	    me.name, me.desc);
	if (strchr (rs.mode, 'o'))
	    send_cmd (me.name, "313 %s %s :is an IRC Operator", source,
		s_RootServ);
	send_cmd (me.name, "317 %s %s 0 %d :seconds idle, signon time",
	    source, s_RootServ, me.since);
    }
    else if ((!stricmp (av[0], s_NickServ)) && nickserv_on == TRUE)
    {
	send_cmd (me.name, "311 %s %s %s %s * :%s", source, s_NickServ,
	    ns.user, ns.host, ns.real);
	send_cmd (me.name, "312 %s %s %s :%s", source, s_NickServ,
	    me.name, me.desc);
	if (strchr (ns.mode, 'o'))
	    send_cmd (me.name, "313 %s %s :is an IRC Operator", source,
		s_NickServ);
	send_cmd (me.name, "317 %s %s 0 %d :seconds idle, signon time",
	    source, s_NickServ, me.since);
    }
    else if ((!stricmp (av[0], s_ChanServ)) && chanserv_on == TRUE)
    {
	send_cmd (me.name, "311 %s %s %s %s * :%s", source, s_ChanServ,
	    cs.user, cs.host, cs.real);
	send_cmd (me.name, "312 %s %s %s :%s", source, s_ChanServ,
	    me.name, me.desc);
	if (strchr (ms.mode, 'o'))
	    send_cmd (me.name, "313 %s %s :is an IRC Operator", source,
		s_ChanServ);
	send_cmd (me.name, "317 %s %s 0 %d :seconds idle, signon time",
	    source, s_ChanServ, me.since);
    }
    else if ((!stricmp (av[0], s_MemoServ)) && memoserv_on == TRUE)
    {
	send_cmd (me.name, "311 %s %s %s %s * :%s", source, s_MemoServ,
	    ms.user, ms.host, ms.real);
	send_cmd (me.name, "312 %s %s %s :%s", source, s_MemoServ,
	    me.name, me.desc);
	if (strchr (ms.mode, 'o'))
	    send_cmd (me.name, "313 %s %s :is an IRC Operator", source,
		s_MemoServ);
	send_cmd (me.name, "317 %s %s 0 %d :seconds idle, signon time",
	    source, s_MemoServ, me.since);
    }
    else if ((!stricmp (av[0], s_GlobalNoticer)) && globalnoticer_on == TRUE)
    {
	send_cmd (me.name, "311 %s %s %s %s * :%s", source, s_GlobalNoticer,
	    gn.user, gn.host, gn.real);
	send_cmd (me.name, "312 %s %s %s :%s", source, s_GlobalNoticer,
	    me.name, me.desc);
	if (strchr (gn.mode, 'o'))
	    send_cmd (me.name, "313 %s %s :is an IRC Operator", source,
		s_GlobalNoticer);
	send_cmd (me.name, "317 %s %s 0 %d :seconds idle, signon time",
	    source, s_GlobalNoticer, me.since);
    }
    else
	send_cmd (me.name, "401 %s %s :No such nick/channel", source, av[0]);
    send_cmd (me.name, "318 %s %s :End of /WHOIS list", source, av[0]);
}

/* Handle a SERVER command */
void m_server (char *source, int8 ac, char **av)
{
    Server *server;

    /* Allocate Server structure and fill it in. */
    server = new_server (av[0]);

    server->uplink = sstrdup (source);

    /* Make note of our uplink for anti-jupe purposes */
    if (atoi (av[1]) == 1)
	memcpy (me.ruplink, av[1], sizeof (me.ruplink));
}

/* Handle an SQUIT command */
void m_squit (char *source, int8 ac, char **av)
{
    User *u;
    Server *s;
    int32 i, delncnt = 0, delscnt = 0;
    char **delnicks = NULL, **delserv = NULL;

    s = servlist[SVHASH(av[0])];

    if (s)
    {
	/* Add all the users for that server into a buffer. */
	for (u = firstuser (); u; u = nextuser ())
	{
	    if (u && !stricmp (s->name, u->server))
	    {
		++delncnt;
		delnicks = srealloc (delnicks, sizeof (char *) *delncnt);
		delnicks[delncnt - 1] = sstrdup (u->nick);
	    }
	}

	/* Add the splitting server into a buffer */
	++delscnt;
	delserv = srealloc (delserv, sizeof (char *) *delscnt);
	delserv[delscnt - 1] = sstrdup (s->name);
    }

    /* Bahamut, Prometheus and Unreal 3.2 only send SQUITs for the splitting server,
       not servers linked to it. So we have to free those servers and their users here.
     */
    if (ircdtype == BAHAMUT || ircdtype == PROMETHEUS || ircdtype == UNREAL3_2)
    {
	for (s = firstserv (); s; s = nextserv ())
	{
	    if (s && !stricmp (s->uplink, av[0]))
	    {
		for (u = firstuser (); u; u = nextuser ())
		{
		    if (u && !stricmp (s->name, u->server))
		    {
			++delncnt;
			delnicks = srealloc (delnicks, sizeof (char *) *delncnt);
			delnicks[delncnt - 1] = sstrdup (u->nick);
		    }
		}

		++delscnt;
		delserv = srealloc (delserv, sizeof (char *) *delscnt);
		delserv[delscnt - 1] = sstrdup (s->name);
	    }
	}
    }

    /* Now we have two buffers, one with the users to delete, and one with servers.
       While we delete the users, we'll check if we're waiting for one of their
       nicknames.
     */
    for (i = 0; i < delncnt; delnicks++, i++)
    {
	u = finduser (*delnicks);

	if (u)
	{
	    if (rs.wantnick && !stricmp (u->nick, rs.nick))
	    {
		send_cmd (s_RootServ, "%s %s", me.token ? "&" : "NICK", rs.nick);
		strscpy (s_RootServ, rs.nick, sizeof (s_RootServ));
		rs.wantnick = 0;
	    }

	    if (ns.wantnick && !stricmp (u->nick, ns.nick))
	    {
		send_cmd (s_NickServ, "%s %s", me.token ? "&" : "NICK", ns.nick);
		strscpy (s_NickServ, ns.nick, sizeof (s_NickServ));
		ns.wantnick = 0;
	    }

	    if (cs.wantnick && !stricmp (u->nick, cs.nick))
	    {
		send_cmd (s_ChanServ, "%s %s", me.token ? "&" : "NICK", cs.nick);
		strscpy (s_ChanServ, cs.nick, sizeof (s_ChanServ));
		cs.wantnick = 0;
	    }

	    if (ms.wantnick && !stricmp (u->nick, ms.nick))
	    {
		send_cmd (s_MemoServ, "%s %s", me.token ? "&" : "NICK", ms.nick);
		strscpy (s_MemoServ, ms.nick, sizeof (s_MemoServ));
		ms.wantnick = 0;
	    }

	    if (gn.wantnick && !stricmp (u->nick, gn.nick))
	    {
		send_cmd (s_GlobalNoticer, "%s %s", me.token ? "&" : "NICK", gn.nick);
		strscpy (s_GlobalNoticer, gn.nick, sizeof (s_GlobalNoticer));
		gn.wantnick = 0;
	    }

	    delete_user (u);
	}

	free (*delnicks);
    }
            
    /* Now free all the affected servers. */
    for (i = 0; i < delscnt; delserv++, i++)
    {
	s = findserv (*delserv);

	if (s)
	    delete_server (s);

	free (*delserv);
    }

    delserv = NULL;
}

/* Handle a KILL command. */
void m_kill (char *source, int8 ac, char **av)
{
    if (ac != 2)
	return;

    /* Before we reintroduce the user, we'll check to see if they've
       been collided more than 5 times in the last 5 seconds. This is
       a common problem with people who forget to rename/disable services
       with the same nicks as some of ours. If they HAVE been collided
       that many times, we'll give them a nick like RootServ5671 and
       send a globops about it. We'll then watch for a nickchange/quit from
       the client with our nick, and retake it if the opportunity presents
       itself.
     */
    if ((rootserv_on == TRUE) && (!stricmp (av[0], s_RootServ)))
    {
	/* How many times we've been collided */
	rs.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((rs.lastcoll >= (time (NULL) - 5)) && (rs.collided >= 5)))
	{
	    /* Check if it's a server killing us. If it's not, then it's
	       likely a regular kill, and we don't want to take a waiting
	       nick.
	     */
	    if (!strchr (source, '.'))
	    {
		/* It's not a server. We'll reintro with the regular
		   nick and leave it at that.
		 */
		intro_user (s_RootServ, rs.user, rs.host, rs.real, rs.mode);
		return;
	    }

	    if (!rs.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (rs.nick, s_RootServ, sizeof (rs.nick));
	    }

	    /* Make note that we want it */
	    rs.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_RootServ, sizeof (s_RootServ), "%s%d", rs.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_RootServ, rs.user, rs.host, rs.real, rs.mode);

	    /* Let the staff know about it */
	    globops (s_RootServ, RPL_COLLISIONS, rs.nick);

	    rs.collided = 0;

	    return;
	}
	else
	    rs.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_RootServ, rs.user, rs.host, rs.real, rs.mode);
    }
    else if ((nickserv_on == TRUE) && !stricmp (av[0], s_NickServ))
    {
	/* How many times we've been collided */
	ns.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((ns.lastcoll >= (time (NULL) - 5)) && (ns.collided >= 5)))
	{
	    /* Check if it's a server killing us. If it's not, then it's
	       likely a regular kill, and we don't want to take a waiting
	       nick.
	     */
	    if (!strchr (source, '.'))
	    {
		/* It's not a server. We'll reintro with the regular
		   nick and leave it at that.
		 */
		intro_user (s_NickServ, ns.user, ns.host, ns.real, ns.mode);
		return;
	    }

	    if (!ns.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (ns.nick, s_NickServ, sizeof (ns.nick));
	    }

	    /* Make note that we want it */
	    ns.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_NickServ, sizeof (s_NickServ), "%s%d", ns.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_NickServ, ns.user, ns.host, ns.real, ns.mode);

	    /* Let the staff know about it */
	    globops (s_NickServ, RPL_COLLISIONS, ns.nick);

	    ns.collided = 0;

	    return;
	}
	else
	    ns.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_NickServ, ns.user, ns.host, ns.real, ns.mode);
    }
    else if ((chanserv_on == TRUE) && !stricmp (av[0], s_ChanServ))
    {
	/* How many times we've been collided */
	cs.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((cs.lastcoll >= (time (NULL) - 5)) && (cs.collided >= 5)))
	{
	    /* Check if it's a server killing us. If it's not, then it's
	       likely a regular kill, and we don't want to take a waiting
	       nick.
	     */
	    if (!strchr (source, '.'))
	    {
		/* It's not a server. We'll reintro with the regular
		   nick and leave it at that.
		 */
		intro_user (s_ChanServ, cs.user, cs.host, cs.real, cs.mode);
		return;
	    }

	    if (!cs.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (cs.nick, s_ChanServ, sizeof (cs.nick));
	    }

	    /* Make note that we want it */
	    cs.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_ChanServ, sizeof (s_ChanServ), "%s%d", cs.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_ChanServ, cs.user, cs.host, cs.real, cs.mode);

	    /* Let the staff know about it */
	    globops (s_ChanServ, RPL_COLLISIONS, cs.nick);

	    cs.collided = 0;

	    return;
	}
	else
	    cs.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_ChanServ, cs.user, cs.host, cs.real, cs.mode);
    }
    else if ((memoserv_on == TRUE) && !stricmp (av[0], s_MemoServ))
    {
	/* How many times we've been collided */
	ms.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((ms.lastcoll >= (time (NULL) - 5)) && (ms.collided >= 5)))
	{
	    /* Check if it's a server killing us. If it's not, then it's
	       likely a regular kill, and we don't want to take a waiting
	       nick.
	     */
	    if (!strchr (source, '.'))
	    {
		/* It's not a server. We'll reintro with the regular
		   nick and leave it at that.
		 */
		intro_user (s_MemoServ, ms.user, ms.host, ms.real, ms.mode);
		return;
	    }

	    if (!ms.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (ms.nick, s_MemoServ, sizeof (ms.nick));
	    }

	    /* Make note that we want it */
	    ms.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_MemoServ, sizeof (s_MemoServ), "%s%d", ms.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_MemoServ, ms.user, ms.host, ms.real, ms.mode);

	    /* Let the staff know about it */
	    globops (s_MemoServ, RPL_COLLISIONS, ms.nick);

	    ms.collided = 0;

	    return;
	}
	else
	    ms.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_MemoServ, ms.user, ms.host, ms.real, ms.mode);
    }
    else if ((globalnoticer_on == TRUE) && !stricmp (av[0], s_GlobalNoticer))
    {
	/* How many times we've been collided */
	gn.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((gn.lastcoll >= (time (NULL) - 5)) && (gn.collided >= 5)))
	{
	    /* Check if it's a server killing us. If it's not, then it's
	       likely a regular kill, and we don't want to take a waiting
	       nick.
	     */
	    if (!strchr (source, '.'))
	    {
		/* It's not a server. We'll reintro with the regular
		   nick and leave it at that.
		 */
		intro_user (s_GlobalNoticer, gn.user, gn.host, gn.real,
		    gn.mode);
		return;
	    }

	    if (!gn.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (gn.nick, s_GlobalNoticer, sizeof (gn.nick));
	    }

	    /* Make note that we want it */
	    gn.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_GlobalNoticer, sizeof (s_GlobalNoticer), "%s%d",
		gn.nick, getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_GlobalNoticer, gn.user, gn.host, gn.real, gn.mode);

	    /* Let the staff know about it */
	    globops (s_GlobalNoticer, RPL_COLLISIONS, gn.nick);

	    gn.collided = 0;

	    return;
	}
	else
	    gn.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_GlobalNoticer, gn.user, gn.host, gn.real, gn.mode);
    }
    else
	do_kill (source, ac, av);
}

/* Handle a nick collision */
void m_collide (char *source, int8 ac, char **av)
{
    /* Before we reintroduce the user, we'll check to see if they've
       been collided more than 5 times in the last 5 seconds. This is
       a common problem with people who forget to rename/disable services
       with the same nicks as some of ours. If they HAVE been collided
       that many times, we'll give them a nick like RootServ986662577 and
       send a globops about it. We'll then watch for a nickchange/quit from
       the client with our nick, and retake it if the opportunity presents
       itself.
     */
    if ((rootserv_on == TRUE) && (!stricmp (av[1], s_RootServ)))
    {
	/* How many times we've been collided */
	rs.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((rs.lastcoll >= (time (NULL) - 5)) && (rs.collided >= 5)))
	{
	    if (!rs.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (rs.nick, s_RootServ, sizeof (rs.nick));
	    }

	    /* Make note that we want it */
	    rs.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_RootServ, sizeof (s_RootServ), "%s%d", rs.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_RootServ, rs.user, rs.host, rs.real, rs.mode);

	    /* Let the staff know about it */
	    globops (s_RootServ, RPL_COLLISIONS, rs.nick);

	    rs.collided = 0;

	    return;
	}
	else
	    rs.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_RootServ, rs.user, rs.host, rs.real, rs.mode);
    }
    else if ((nickserv_on == TRUE) && !stricmp (av[1], s_NickServ))
    {
	/* How many times we've been collided */
	ns.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((ns.lastcoll >= (time (NULL) - 5)) && (ns.collided >= 5)))
	{
	    if (!ns.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (ns.nick, s_NickServ, sizeof (ns.nick));
	    }

	    /* Make note that we want it */
	    ns.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_NickServ, sizeof (s_NickServ), "%s%d", ns.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_NickServ, ns.user, ns.host, ns.real, ns.mode);

	    /* Let the staff know about it */
	    globops (s_NickServ, RPL_COLLISIONS, ns.nick);

	    ns.collided = 0;

	    return;
	}
	else
	    ns.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_NickServ, ns.user, ns.host, ns.real, ns.mode);
    }
    else if ((chanserv_on == TRUE) && !stricmp (av[1], s_ChanServ))
    {
	/* How many times we've been collided */
	cs.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((cs.lastcoll >= (time (NULL) - 5)) && (cs.collided >= 5)))
	{
	    if (!cs.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (cs.nick, s_ChanServ, sizeof (cs.nick));
	    }

	    /* Make note that we want it */
	    cs.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_ChanServ, sizeof (s_ChanServ), "%s%d", cs.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_ChanServ, cs.user, cs.host, cs.real, cs.mode);

	    /* Let the staff know about it */
	    globops (s_ChanServ, RPL_COLLISIONS, cs.nick);

	    cs.collided = 0;

	    return;
	}
	else
	    cs.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_ChanServ, cs.user, cs.host, cs.real, cs.mode);
    }
    else if ((memoserv_on == TRUE) && !stricmp (av[1], s_MemoServ))
    {
	/* How many times we've been collided */
	ms.collided++;

	/* If this is true, we give up trying to retake our nick.. */
	if (((ms.lastcoll >= (time (NULL) - 5)) && (ms.collided >= 5)))
	{
	    if (!ms.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (ms.nick, s_MemoServ, sizeof (ms.nick));
	    }

	    /* Make note that we want it */
	    ms.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_MemoServ, sizeof (s_MemoServ), "%s%d", ms.nick,
		getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_MemoServ, ms.user, ms.host, ms.real, ms.mode);

	    /* Let the staff know about it */
	    globops (s_MemoServ, RPL_COLLISIONS, ms.nick);

	    ms.collided = 0;

	    return;
	}
	else
	    ms.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_MemoServ, ms.user, ms.host, ms.real, ms.mode);
    }
    else if ((globalnoticer_on == TRUE) && !stricmp (av[1], s_GlobalNoticer))
    {
	/* How many times we've been collided */
	++gn.collided;

	/* If this is true, we give up trying to retake our nick.. */
	if (((gn.lastcoll >= (time (NULL) - 5)) && (gn.collided >= 5)))
	{
	    if (!gn.wantnick)
	    {
		/* Make note of the nick we want */
		strscpy (gn.nick, s_GlobalNoticer, sizeof (gn.nick));
	    }

	    /* Make note that we want it */
	    gn.wantnick = 1;

	    /* Give ourselves a new nick */
	    snprintf (s_GlobalNoticer, sizeof (s_GlobalNoticer), "%s%d",
		gn.nick, getpid ());

	    /* Introduce with the new nick */
	    intro_user (s_GlobalNoticer, gn.user, gn.host, gn.real, gn.mode);

	    /* Let the staff know about it */
	    globops (s_GlobalNoticer, RPL_COLLISIONS, gn.nick);

	    gn.collided = 0;

	    return;
	}
	else
	    gn.lastcoll = time (NULL);

	/* Re-introduce our client */
	intro_user (s_GlobalNoticer, gn.user, gn.host, gn.real, gn.mode);
    }
}

/* Handle a PROTOCTL command */
void m_protoctl (char *source, int8 ac, char **av)
{
    int i, token = 0;
    int nickv2 = 0, umode2 = 0;

    for (i = 0; i < ac; i++)
    {
	if (!stricmp (av[i], "TOKEN"))
	    token = 1;

	if (!stricmp (av[i], "NICKv2"))
	    nickv2 = 1;

	if (!stricmp (av[i], "UMODE2"))
	    umode2 = 1;
    }

    if (token)
	me.token = 1;

    if (ircdtype == UNREAL3 || ircdtype == UNREAL3_2)
    {
	if (!nickv2 || !umode2)
	{
	    send_cmd (NULL,
		"ERROR :Incompatible IRCd (I'm configured to support Unreal 3.1+)");
	    snprintf (quitmsg, BUFSIZE,
		"Incompatible IRCd (I'm configured to support Unreal 3.1+)");
	    runflags |= RUN_SHUTDOWN;
	}
    }
}

/* Handle a MODE command */
void m_mode (char *source, int8 ac, char **av)
{
    if (*av[0] == '#')
    {
	if (ac < 2)
	    return;

	do_cmode (source, ac, av);
    }
    else
    {
	if (ac != 2)
	    return;

	do_umode (source, ac, av);
    }
}

/* Handle an UnrealIRCd UMODE2 command */
void m_umode2 (char *source, int8 ac, char **av)
{
    av[1] = sstrdup (av[0]);
    strcpy (av[0], source);

    do_umode (source, ac, av);
}

/* Change the users fake hostname */
void m_sethost (char *source, int8 ac, char **av)
{
    User *u;

    if (ac != 1)
	return;

    u = finduser (source);

    if (!u)
	return;

    free (u->host);

    u->host = sstrdup (av[0]);
}

/* CHGHOST. Similar to SETHOST, use the same function. */
void m_chghost (char *source, int8 ac, char **av)
{
    if (ac == 2)
	m_sethost (av[0], ac-1, av+1);
}

/* Change a users ident (aka username) */
void m_setident (char *source, int8 ac, char **av)
{
    User *u;

    if (ac != 1)
	return;

    u = finduser (source);

    if (!u)
	return;

    free (u->user);

    u->user = sstrdup (av[0]);
}

/* CHGIDENT. Similar to SETIDENT, use the same function. */
void m_chgident (char *source, int8 ac, char **av)
{
    if (ac == 2)
	m_setident (av[0], ac-1, av+1);
}

/* Change a users realname. */
void m_setname (char *source, int8 ac, char **av)
{
    User *u;

    if (ac != 1)
	return;

    u = finduser (source);

    if (!u)
        return;

    free (u->real);

    u->real = sstrdup (av[0]);
}

/* CHGNAME. Similar to SETNAME, use the same function. */
void m_chgname (char *source, int8 ac, char **av)
{
    if (ac == 2)
	m_setname (av[0], ac-1, av+1);
}

/* Handle the ENDBURST command */
void m_endburst (char *source, int8 ac, char **av)
{
    globops (s_RootServ, "Synched to network in %s: %d users, %d channels, %d servers.",
	duration (time (NULL) - burststart, 1), userstat, chancnt, servcnt + 1);

    bursting = 0;

#ifdef HAVE_GETTIMEOFDAY
    gettimeofday (&burstnow, NULL);
    timersub (&burstnow, &burstms, &bursttmp);

    if (tv2ms (&bursttmp) > 1000)
	log_sameline (0, "completed in %s.", duration (tv2ms (&bursttmp) / 1000, 1));
    else
	log_sameline (0, "completed in %d ms.", tv2ms (&bursttmp));
#endif

    /* Now that the burst is done, run through every channel, and do the modechanges
       and topic changes as needed.
     */
    check_eb ();
}

/* Handle PRIVMSGs. */
void m_privmsg (char *source, int8 ac, char **av)
{
    User *u;
    char buf[BUFSIZE], text[BUFSIZE];

    if (ac != 2)
	return;

    u = finduser (source);

    if (!u)
    {
	log ("No user record found for %s!", source);
	notice (s_RootServ, source, RPL_YOU_DONT_EXIST);
	notice (s_RootServ, source, RPL_PLEASE_RECONNECT);
	return;
    }

    /* Run it through flood checks */
    if (floodmsgs)
    {
	time_t now = time (NULL);

	/* If it's a channel PRIVMSG, ignore it. We have nothing to do with
	   channels.
	 */
	if (av[0][0] == '#')
	    return;

	/* Check if they're being ignored. */
	if (u->offences > 10)
	{
	    /* They're being ignored. Check if the ignore expired. */
	    if (now - u->lastmsg > 30)
	    {
		/* They're not being ignored anymore. */
		u->offences = u->offences - 10;
		u->lastmsg = now;
		u->msgs = 0;
	    }
	    else
		return;
	}

	if (now - u->lastmsg > floodtime)
	{
	    u->lastmsg = now;
	    u->msgs = 0;
	}

	u->msgs++;

	if (u->msgs > floodmsgs)
	{
	    /* They set off the trigger. Take action. */
	    if (!u->offences)
	    {
		/* First ignore. */
		u->lastmsg = now;
		u->msgs = 0;

		/* So we know we're ignoring them */
		u->offences = 11;

		/* Tell them. The stern voice of RootServ should be
		   an attention grabber. <g>
		 */
		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGERED);
		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGER, floodmsgs,
		    duration (floodtime, 2));
		notice (s_RootServ, u->nick, RS_FLOOD_FIRST);

		/* Complain in GlobOps */
		if (noisyflood_on == TRUE)
		    globops (s_RootServ, RS_FLOOD_GLOBOPS, u->nick, "");

		return;
	    }

	    if (u->offences == 1)
	    {
		/* Second ignore. */
		u->lastmsg = now;
		u->msgs = 0;

		/* So we know we're ignoring them */
		u->offences = 12;

		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGERED);
		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGER, floodmsgs,
		    duration (floodtime, 2));
		notice (s_RootServ, u->nick, RS_FLOOD_SECOND);

		/* Complain in GlobOps */
		if (noisyflood_on == TRUE)
		    globops (s_RootServ, RS_FLOOD_GLOBOPS, u->nick, "");

		return;
	    }

	    if (u->offences == 2)
	    {
		AutoKill *akill;
		char mask[BUFSIZE], *username, *hostname, msgbuf[31];

		/* Third offence. AKill. */
		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGERED);
		notice (s_RootServ, u->nick, RS_FLOOD_TRIGGER, floodmsgs,
		    duration (floodtime, 2));

		/* Complain in GlobOps */
		if (noisyflood_on == TRUE)
		    globops (s_RootServ, RS_FLOOD_GLOBOPS, u->nick, " (Killed)");

		snprintf (msgbuf, sizeof (msgbuf), "%s", av[1]);

		log ("RS:FLOOD:KILL: %s!%s@%s (%s%s)", u->nick, u->user, u->host, msgbuf,
		   strlen (msgbuf) == 30 ? "..." : "");

		u->offences = 13;

		snprintf (mask, sizeof (mask), "*@%s", u->host);

		if (akillcnt >= akill_size)
		{
		    if (akill_size < 8)
			akill_size = 8;
		    else if (akill_size >= 16384)
			akill_size = 32767;
		    else
			akill_size *= 2;

		    akills = srealloc (akills, sizeof (*akills) * akill_size);
		}

		akill = &akills[akillcnt];
		akill->mask = sstrdup (mask);
		akill->setter = sstrdup (s_RootServ);
		akill->realname = 0;
		akill->set = now;
		akill->expires = 1800;
		akill->reason = sstrdup (RS_AKILL_FLOOD);

		akillcnt++;

		username = sstrdup (akill->mask);
		hostname = strchr (username, '@');

		if (!hostname)
		    return;

		*hostname++ = 0;

		if (ircdtype == BAHAMUT || ircdtype == PROMETHEUS)
		    send_cmd (me.name, "AKILL %s %s 0 %s %ld :%s", hostname,
			username, s_RootServ, time (NULL), akill->reason);
		else
		    send_cmd (me.name, "%s %s %s :%s", me.token ? "V" : "AKILL",
			hostname, username, akill->reason);

		free (username);

		return;
	    }
	}
    }

    /* RootServ? */
    if (stristr (av[0], s_RootServ))
    {
	/* What a nick@server for RootServ looks like */
	snprintf (buf, sizeof (buf), "%s@%s", s_RootServ, me.name);

	/* Do our checks on a copy of the original */
	snprintf (text, sizeof (text), "%s", av[1]);

	/* Check if we should deny a message via security settings */
	if (stricmp (av[0], buf) && ((securitysetting == 1 ||
	    (securitysetting == 2 && ispass (text)))
	    && !isctcp (text)))
	{
	    /* Tell them to use nick@server and bail */
	    if (haveserv_on == TRUE)
		notice (s_RootServ, u->nick, RPL_SECURE_MESSAGES, "",
		    s_RootServ, "", "");
	    else
		notice (s_RootServ, u->nick, RPL_SECURE_MESSAGES, "MSG ",
		    s_RootServ, "@", me.name);
	    return;
	}
	else
	{
	    /* Passed security checks, throw it through RootServ */
	    rootserv (u, av[1]);
	    return;
	}
    }

    /* NickServ? */
    if (stristr (av[0], s_NickServ))
    {
	/* What a nick@server for NickServ looks like */
	snprintf (buf, sizeof (buf), "%s@%s", s_NickServ, me.name);

	/* Do our checks on a copy of the original */
	snprintf (text, sizeof (text), "%s", av[1]);

	/* Check if we should deny a message via security settings */
	if (stricmp (av[0], buf) && ((securitysetting == 1 ||
	    (securitysetting == 2 && ispass (text)))
	    && !isctcp (text)))
	{
	    /* Tell them to use nick@server and bail */
	    if (haveserv_on == TRUE)
		notice (s_NickServ, u->nick, RPL_SECURE_MESSAGES, "",
		    s_NickServ, "", "");
	    else
		notice (s_NickServ, u->nick, RPL_SECURE_MESSAGES, "MSG ",
		    s_NickServ, "@", me.name);
	    return;
	}
	else
	{
	    /* Passed security checks, throw it through NickServ */
	    nickserv (u, av[1]);
	    return;
	}
    }

    /* ChanServ? */
    if (stristr (av[0], s_ChanServ))
    {
	/* What a nick@server for ChanServ looks like */
	snprintf (buf, sizeof (buf), "%s@%s", s_ChanServ, me.name);

	/* Do our checks on a copy of the original */
	snprintf (text, sizeof (text), "%s", av[1]);

	/* Check if we should deny a message via security settings */
	if (stricmp (av[0], buf) && ((securitysetting == 1 ||
	    (securitysetting == 2 && ispass (text)))
	    && !isctcp (text)))
	{
	    /* Tell them to use nick@server and bail */
	    if (haveserv_on == TRUE)
		notice (s_ChanServ, u->nick, RPL_SECURE_MESSAGES, "",
		    s_ChanServ, "", "");
	    else
		notice (s_ChanServ, u->nick, RPL_SECURE_MESSAGES, "MSG ",
		    s_ChanServ, "@", me.name);
	    return;
	}
	else
	{
	    /* Passed security checks, throw it through ChanServ */
	    chanserv (u, av[1]);
	    return;
	}
    }

    /* MemoServ? */
    if (stristr (av[0], s_MemoServ))
    {
	/* What a nick@server for MemoServ looks like */
	snprintf (buf, sizeof (buf), "%s@%s", s_MemoServ, me.name);

	/* Check if we should deny a message via security settings */
	if (stricmp (av[0], buf) && (securitysetting == 1 &&
	    !isctcp (av[1])))
	{
	    /* Tell them to use nick@server and bail */
	    if (haveserv_on == TRUE)
		notice (s_MemoServ, u->nick, RPL_SECURE_MESSAGES, "",
		    s_MemoServ, "", "");
	    else
		notice (s_MemoServ, u->nick, RPL_SECURE_MESSAGES, "MSG ",
		    s_MemoServ, "@", me.name);
	    return;
	}
	else
	{
	    /* Passed security checks, throw it through MemoServ */
	    memoserv (u, av[1]);
	    return;
	}
    }

    snprintf (buf, sizeof (buf), "%s@%s", s_GlobalNoticer, me.name);
    if (!stricmp (av[0], s_GlobalNoticer) || !stricmp (av[0], buf))
    {
	globalnoticer (u, av[1]);
	return;
    }
}

Message messages[] = {

    /* These commands are ignored. */
    {"NOTICE",	NULL},
    {"GNOTICE",	NULL},
    {"PASS",	NULL},
    {"CAPAB",	NULL},
    {"SVINFO",	NULL},

    /* These commands are used by every IRCd we support. They're ordered
       by frequency of use to speed up processing.
     */
    {"PRIVMSG",	m_privmsg},
    {"NICK",	m_nick},
    {"MODE",	m_mode},
    {"SJOIN",	m_sjoin},
    {"JOIN",	m_join},
    {"PART",	m_part},
    {"KICK",	m_kick},
    {"AWAY",	m_away},
    {"TOPIC",	m_topic},
    {"QUIT",	m_quit},
    {"PING",	m_ping},
    {"KILL",	m_kill},
    {"UMODE2",	m_umode2},
    {"SETIDENT",m_setident},
    {"SETHOST",	m_sethost},
    {"SETNAME", m_setname},
    {"CHGHOST",	m_chghost},
    {"CHGIDENT",m_chgident},
    {"CHGNAME",	m_chgname},
    {"VERSION",	m_version},
    {"INFO",	m_info},
    {"WHOIS",	m_whois},
    {"STATS",	m_stats},
    {"MOTD",	m_motd},
    {"TIME",	m_time},
    {"351",	m_vreply},
    {"ADMIN",	m_admin},
    {"436",	m_collide},
    {"SQUIT",	m_squit},
    {"NETINFO",	m_netinfo},
    {"SERVER",	m_server},
    {"PROTOCTL",m_protoctl},
    {"EB",	m_endburst},
    {"ERROR",	m_error},

    /* These commands are ignored. */
    {"B",	NULL},
    {"Z",	NULL},
    {"<",	NULL},

    /* Tokens - DreamForge and Unreal support these tokens. */
    {"!",	m_privmsg},
    {"&",	m_nick},
    {"G",	m_mode},
    {"~",	m_sjoin},
    {"C",	m_join},
    {"D",	m_part},
    {"H",	m_kick},
    {"6",	m_away},
    {")",	m_topic},
    {",",	m_quit},
    {"8",	m_ping},
    {".",	m_kill},
    {"|",	m_umode2},
    {"AD",	m_setident},
    {"AA",	m_sethost},
    {"AE",	m_setname},
    {"AL",	m_chghost},
    {"AZ",	m_chgident},
    {"BK",	m_chgname},
    {"+",	m_version},
    {"/",	m_info},
    {"#",	m_whois},
    {"2",	m_stats},
    {"F",	m_motd},
    {">",	m_time},
    {"@",	m_admin},
    {"-",	m_squit},
    {"AO",	m_netinfo},
    {"'",	m_server},
    {"_",	m_protoctl},
    {"5",	m_error},
    {NULL}
};

Message *find_message (const char *name)
{
    Message *m;

    for (m = messages; m->name; m++)
	if (!stricmp (name, m->name))
	    return m;

    return NULL;
}
